// Copyright 2014 Google Inc. All Rights Reserved.

#include "NavigationStatusEndpoint.h"

void NavigationStatusEndpoint::addDiscoveryInfo(ServiceDiscoveryResponse* sdr) {
    Service* service = sdr->add_services();
    service->set_id(id());
    NavigationStatusService* ics = service->mutable_navigation_status_service();
    ics->set_minimum_interval_ms(mMinIntervalMs);
    ics->set_type(mType);
    if (mType == NavigationStatusService_InstrumentClusterType_IMAGE) {
      NavigationStatusService_ImageOptions* io = ics->mutable_image_options();
      io->set_height(mHeight);
      io->set_width(mWidth);
      io->set_colour_depth_bits(mColourDepthBits);
    }
}

int NavigationStatusEndpoint::routeMessage(
        uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg) {
    int ret = STATUS_UNEXPECTED_MESSAGE;
    uint8_t* ptr = (uint8_t*)msg->raw() + sizeof(uint16_t);
    size_t len = msg->size() - sizeof(uint16_t);

    switch (type) {
        case INSTRUMENT_CLUSTER_NAVIGATION_STATUS: {
            NavigationStatus status;
            if (PARSE_PROTO(status, ptr, len)) {
                handleNavigationStatus(status);
                ret = STATUS_SUCCESS;
            }
            break;
        }
        case INSTRUMENT_CLUSTER_NAVIGATION_TURN_EVENT: {
            NavigationNextTurnEvent event;
            if (PARSE_PROTO(event, ptr, len)) {
                handleNavigationNextTurnEvent(event);
                ret = STATUS_SUCCESS;
            }
            break;
        }
        case INSTRUMENT_CLUSTER_NAVIGATION_DISTANCE_EVENT: {
            NavigationNextTurnDistanceEvent event;
            if (PARSE_PROTO(event, ptr, len)) {
                handleNavigationDistanceEvent(event);
                ret = STATUS_SUCCESS;
            }
            break;
        }
    }
    return ret;
}

void NavigationStatusEndpoint::start() {
    NavigationStatusStart start;
    IoBuffer buf;
    mRouter->marshallProto(INSTRUMENT_CLUSTER_START, start, &buf);
    queueOutgoing(buf.raw(), buf.size());
}

void NavigationStatusEndpoint::stop() {
    NavigationStatusStop stop;
    IoBuffer buf;
    mRouter->marshallProto(INSTRUMENT_CLUSTER_STOP, stop, &buf);
    queueOutgoing(buf.raw(), buf.size());
}

void NavigationStatusEndpoint::handleNavigationStatus(const NavigationStatus &status) {
    mCallbacks->navigationStatusCallback(status.status());
}

void NavigationStatusEndpoint::handleNavigationNextTurnEvent(
        const NavigationNextTurnEvent &event) {
    int turnSide = event.has_turn_side() ? event.turn_side() : 0;
    const string& image = event.has_image() ? event.image() : "";
    int turnAngle = event.has_turn_angle() ? event.turn_angle() : -1;
    int turnNumber = event.has_turn_number() ? event.turn_number() : -1;
    mCallbacks->navigationNextTurnCallback(event.road(), turnSide, event.event(), image,
            turnAngle, turnNumber);
}

void NavigationStatusEndpoint::handleNavigationDistanceEvent(
        const NavigationNextTurnDistanceEvent &event) {
    if (event.has_display_distance_unit() && event.display_distance_unit()
            != NavigationNextTurnDistanceEvent_DistanceUnits_UNKNOWN_DISTANCE_UNIT) {
        mCallbacks->navigationNextTurnDistanceCallback(event.distance_meters(),
                event.time_to_turn_seconds(), event.display_distance_e3(),
                event.display_distance_unit());
    }
}
